package hardware;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.Scanner;

import bits.BitString;

/**
 * LoadableMemory
 * Interface for any Memory that can be loaded from a stream of
 * BitString values.
 *
 * Implementation must provide the Scanner version of loading since
 * the File-based versions provide default implementations in terms of
 * the Scanner version.
 */
public interface LoadableMemory {
  /**
   * Load this memory, from the given startAddress, with the 16-bit
   * BitStrings that can be scanned (one per line) by fin.
   * @param fin input scanner from which lines of bits are to be read
   * @param startAddress address in this memory where first word read
   *   is to be stored
   * @return the number of words read from fin
   */
  public short fillFromScanner(Scanner fin, BitString startAddress);

  /**
   * Load this memory (from startAddress) with BitStrings found in
   * fileName.
   * @param fileName path name to a valid hack MACHINE LANGUAGE FILE
   * @param startAddress address in the memory to start filling from
   * @return the number of words read from the file
   * @throws FileNotFoundException if the named file cannot by opened
   *   for input.
   */
  public default short fillFromFile(String fileName, BitString startAddress)
    throws FileNotFoundException {
    return fillFromScanner(new Scanner(new File(fileName)), startAddress);
  }

  /**
   * Load this memory (from address 0) with BitStrings found in fileName.
   *
   * @param fileName     path name to a valid hack MACHINE LANGUAGE FILE
   * @param startAddress address in the memory to start filling from
   * @return the number of words read from the file
   * @throws FileNotFoundException if the named file cannot by opened
   *   for input.
   */
  public default short fillFromFile(String fileName)
    throws FileNotFoundException {
    return fillFromFile(fileName, new BitString()); // fill from 0
  }

}
