package bits;

import java.util.stream.Collectors;
import java.util.Map;
import java.util.Set;

import static java.util.Map.entry;
import static java.util.Map.Entry;

/**
 * InstructionToASM - Convert to/from ASM code string and Instruction object
 */
public class InstructionToASM {
  // map ASM JMP mnemonic to three binary characters
  private static final Map<String, String> JMP2Bits =
    Map.ofEntries(
      entry("",    "000"),
      entry("JGT", "001"),
      entry("JEQ", "010"),
      entry("JGE", "011"),
      entry("JLT", "100"),
      entry("JNE", "101"),
      entry("JLE", "110"),
      entry("JMP", "111")
    );
  // reverse map from three bit characters to ASM JMP code
  private static final Map<String, String> Bits2JMP =
    JMP2Bits.entrySet().stream().
    collect(Collectors.toMap(Entry::getValue, Entry::getKey, (oldValue, newValue) -> oldValue));

  // map ASM DST mnemonic to three binary characters
  private static final Map<String, String> DST2Bits =
    Map.ofEntries(
      entry("",    "000"),
      entry("M",   "001"),
      entry("D",   "010"),
      entry("MD",  "011"),
      entry("A",   "100"),
      entry("AM",  "101"),
      entry("AD",  "110"),
      entry("AMD", "111")
    );
  // reverse map from three bit characters to DST mnemonic
  private static final Map<String, String> Bits2DST =
    DST2Bits.entrySet().stream().
    collect(Collectors.toMap(Entry::getValue, Entry::getKey, (oldValue, newValue) -> oldValue));

  // map known ASM code mnemonics to seven bit strings (acccccc)
  private static final Map<String, String> ASM2Bits =
    Map.ofEntries(
      entry("0",   "0101010"),
      entry("1",   "0111111"),
      entry("-1",  "0111010"),
      entry("D",   "0001100"),
      entry("A",   "0110000"),
      entry("M",   "1110000"),
      entry("!D",  "0001101"),
      entry("!A",  "0110001"),
      entry("!M",  "1110001"),
      entry("-D",  "0001111"),
      entry("-A",  "0110011"),
      entry("-M",  "1110011"),
      entry("D+1", "0011111"),
      entry("A+1", "0110111"),
      entry("M+1", "1110111"),
      entry("D-1", "0001110"),
      entry("A-1", "0110010"),
      entry("M-1", "1110010"),
      entry("D+A", "0000010"),
      entry("D+M", "1000010"),
      entry("D-A", "0010011"),
      entry("D-M", "1010011"),
      entry("A-D", "0000111"),
      entry("M-D", "1000111"),
      entry("D&A", "0000000"),
      entry("D&M", "1000000"),
      entry("D|A", "0010101"),
      entry("D|M", "1010101")
  );
  // reverse map from seven A and COMP bits to ASM mnemoic
  private static final Map<String, String> Bits2ASM =
    ASM2Bits.entrySet().stream().
    collect(Collectors.toMap(Entry::getValue, Entry::getKey, (oldValue, newValue) -> oldValue));

  /**
   * Instruction factory: Takes an ASM instruction mnemonic (no
   * spaces) and returns the matching bit string.
   *
   * @param asm - string containing a valid Hack ASM instruction without
   * whitespace.
   * @return new machine Instruction for the assembly
   * @note NO ERROR CHECKING!
   */
  public static Instruction fromASM(String asm) {
    if (asm.startsWith("@")) {
      return new Instruction(Short.parseShort(asm.substring(1)));
    }
    int equals = asm.indexOf("=");
    String dst = (equals >= 0) ? asm.substring(0, equals) : "";
    asm = (equals >= 0) ? asm.substring(equals+1) : asm;
    int semi = asm.indexOf(";");
    String jmp = (semi >= 0) ? asm.substring(semi+1) : "";
    asm = (semi >= 0) ? asm.substring(0, semi) : asm;

    String bits = "111" + ASM2Bits.get(asm) + DST2Bits.get(dst) + JMP2Bits.get(jmp);

    return new Instruction(bits);
  }


  private static String acomp(String bits) {
    return bits.substring(3,10);
  }

  private static String dst(String bits) {
    return bits.substring(10, 13);
  }

  private static String jmp(String bits) {
    return bits.substring(13, 16);
  }

  private static String CInstructionToASM(Instruction instruction) {
    String bits = instruction.stringOfZeroesAndOnes();

    String retval = Bits2ASM.get(acomp(bits));
    String d = Bits2DST.get(dst(bits));
    if (!d.isEmpty())
      retval = d + "=" + retval;

    String j = Bits2JMP.get(jmp(bits));
    if (!j.isEmpty())
      retval = retval + ";" + j;
    return retval;
  }

  private static String AInstructionToASM(Instruction instruction) {
    return String.format("@%d", instruction.toInt());
  }

  /**
   * toASM - Convert an Instruction to a Hack ASM mnemonic
   * @param instruction - bitstring to convert (actually an instruction)
   * @return a Hack mnemonic with a dst, acomp, and jmp part as necessary
   * @note stringOfZeroesAndOnes is used by this command/helpers
   */
  public static String toASM(Instruction instruction) {
    if (instruction.AInstruction())
      return AInstructionToASM(instruction);
    else
      return CInstructionToASM(instruction);
  }

}
