package hardware;

import java.util.Scanner;

import bits.BitString;
import hardware_interface.Memory;
import util.CSIIILogger;

public class RAM
  implements Memory {
  public static final int DEFAULT_RAM_SIZE = 32768;
  private int memCapacity;
  private BitString [] backingStore;
  private int memSize;

  CSIIILogger logger;

  public RAM(int capacity) {
    memCapacity = capacity;;
    backingStore = new BitString[memCapacity];
    for (int address = 0; address < memCapacity; address++)
      backingStore[address] = new BitString();
    memSize = 0;
    logger = new CSIIILogger();
  }

  public RAM() {
    this(DEFAULT_RAM_SIZE);
  }

  public int capacity() {
    return this.memCapacity;
  }

  private void setSize(int maybeSize) {
    memSize = Math.max(memSize, maybeSize+1);
  }

  public int size() {
    return memSize;
  }

  public void setLogger(CSIIILogger logger) {
    this.logger = logger;
  }

  private int validateAddress(BitString address, int max, String functionName) {
    int nAddress = address.toInt();
    if (nAddress >= max)
      throw new IndexOutOfBoundsException(String.format("%s Bounds Exceeded in %s with %s [%d/%d]",
          "RAM", functionName, address, nAddress, max));
    setSize(nAddress);
    return nAddress;
  }

  private int  validateAddress(BitString address, String functionName) {
    return validateAddress(address, memCapacity, functionName);
  }

  public BitString get(BitString address) {
    int validAddress = validateAddress(address, "get");
    return backingStore[validAddress];
  }

  public void set(BitString address, BitString value) {
    int validAddress = validateAddress(address, "set");
    logger.println(getClass().getName(), "set", String.format("[%s] %s -> %s", address, value, backingStore[validAddress]));
   backingStore[validAddress] = value;
  }

  public short fillFromScanner(Scanner fin, BitString startAddress) {
    int addr = startAddress.toInt();
    short wordsRead = 0;
    while (fin.hasNextLine()) {
      String word = fin.nextLine();
      if (word.isBlank())
        continue;
      backingStore[addr++] = new BitString(word);
      wordsRead++;
    }
    setSize(wordsRead);
    return wordsRead;
  }

  public short[] rangeToShort(BitString first, BitString last) {
    int initMemSize = memSize;
    int nStart = validateAddress(first, "rangeToShort");
    int nLast = validateAddress(last, "rangeToShort");
    int nSize = (short) (nLast - nStart + 1);

    short[] retval = new short[nSize];
    for (int i = 0; i < nSize; i++) {
      retval[i] = backingStore[nStart].toShort();
      nStart++;
    }
    memSize = initMemSize;
    return retval;
  }

  public BitString[] rangeToBitString(BitString first, BitString last) {
    int initMemSize = memSize;
    int nStart = validateAddress(first, "rangeToBitString");
    int nLast = validateAddress(last, "rangeToBitString");
    int nSize = nLast - nStart + 1;

    BitString[] retval = new BitString[nSize];
    for (int i = 0; i < nSize; i++) {
      retval[i] = backingStore[nStart];
      nStart++;
    }
    memSize = initMemSize;
    return retval;
  }
}
