package hardware_interface;

import bits.BitString;

/**
 * Memory - An "array" of *read/write*, random-access bit strings.
 *
 * LoadableHardware - permits the contents to be set from a file.
 * PrintableHardware - permits the main program to request
 * sub-sequences of the contents (for printing, for example).
 */
public interface Memory
  extends LoadableHardware, PrintableHardware {
  /**
   * Number of words in the memory; they have addresses [0..capacity()-1]
   */
  public int capacity();

  /**
   * Number of words in the memory that have been loaded or touched.
   *
   * @return one more than the *highest* address that has been read
   * (get) or written (set). One more to account for the 0 address.
   */
  public int size();

  /**
   * Get the contents of the memory from the given location.
   *
   * Note that the address is unsigned: use BitString.asInt to get the
   * index into the memory (and ignore the sign bit being set from bit
   * 15).
   *
   * @param address - location in the memory to return
   * @return the BitString stored in the given location
   * @throws IndexOutOfBoundsException if address is beyond the bounds
   * (capacity()) of the memory.
   */
  public BitString get(BitString address);

  /**
   * Set the contents of the memory at the given location.
   *
   * Note that the address is unsigned: use BitString.asInt to get the
   * index into the memory (and ignore the sign bit being set from bit
   * 15).
   *
   * @param address - location in the memory to set
   * @param value - BitString to store in the memory
   * @throws IndexOutOfBoundsException if address is beyond the bounds
   *                                   (capacity()) of the memory.
   */
  public void set(BitString address, BitString vaule);

}
