package simulation;

import java.util.Scanner;
import java.io.FileNotFoundException;
import bits.BitString;
import bits.Instruction;
import bits.InstructionToASM;
import hardware_interface.N2TCPU;
import hardware_interface.Memory;
import hardware_interface.InstructionMemory;
import hardware.RAM;
import hardware.ROM;
import hardware.CPU;


/**
 * CPUClient - Processes command-line parameters, creates ROM, RAM,
 * and CPU, then loops, executing w/ CPU until the machine halts.
 */
public class CPUClient {
  // useful (?) BitString constants.
  public static final BitString one = new BitString((short) 1);
  public static final BitString zero = new BitString();
  public static final BitString negativeOne = new BitString((short) -1);

  // file to read for an input Hack file
  private String inputFileName;
  // print CPU state every how many clock cycles;
  // negative value turns printing off
  private int cpuStateCount;

  /**
   * Initialize the fields of a CPU Client object.
   */
  public CPUClient(String fname, int cpu) {
    inputFileName = fname;
    cpuStateCount = cpu;
  }


  /**
   * Main program: process command-line arguments, construct new
   * client, call run.
   */
  public static void main(String[] args) {
    int cpuStateCount = -1;
    String cpuStateString = "--cpu";

    String fname = "";

    for (int i = 0; i < args.length; i++) {
      String argument = args[i];
      if (argument.equals(cpuStateString)) {
        String cpu = args[i+1];
        cpuStateCount = Integer.parseInt(cpu);
        i++;
      } else {
        fname = argument;
      }
    }

    // terminate without a file name argument
    if (fname.isEmpty()) {
      usage();
      System.exit(1);
    }

    CPUClient cpu = new CPUClient(fname, cpuStateCount);
    cpu.run();
  }

  /**
   * Show usage message
   */
  public static void usage() {
    System.err.println("usage: CPUClient <hackFileName> [--cpu <n>]");
    System.err.println("       where");
    System.err.println("         <hackFileName> - path to a Hack machine language file");
    System.err.println("                          REQUIRED");
    System.err.println("         --cpu <n> - print CPU state every <n> clock cycles;");
    System.err.println("                     optional, defaults to off");
  }

  /**
   * Create the ROM, the RAM, and CPU. Execute the Hack program to end.
   *
   */
  public void run() {
    InstructionMemory theROM = new ROM();

    try {
    theROM.fillFromFile(inputFileName);
    } catch (FileNotFoundException fnfe) {
      System.err.println(String.format("Cannot open %s for input.", inputFileName));
      System.exit(2);
    }

    System.out.println("Loaded Program:");
    System.out.println("   +-----------------------------------+");
    displayProgram(theROM);
    System.out.println("   +-----------------------------------+");
    System.out.println();

    // auto loaded with 0's
    Memory theRAM = new RAM();

    N2TCPU cpu = new CPU(theRAM, theROM);

    int clock = 0;

    if (cpuStateCount > 0) {
      System.out.println(String.format("Before clock cycle %d:", clock));
      System.out.println(cpu.state());
    }

    // fetch first so the machine could halt, immediately
    cpu.fetch();
    while(!cpu.halt()) {
      cpu.decode();
      cpu.execute();
      clock++;
      if ((cpuStateCount > 0) && (clock % cpuStateCount == 0)) {
        System.out.println(String.format("Before clock cycle %d:", clock));
        //System.out.println(cpu.state());
        System.out.println("   +-----------------------------------+");
        displayData(theRAM);
        System.out.println("   +-----------------------------------+");
      }
      cpu.fetch();
    }

    System.out.println("Modified RAM:");
    System.out.println("   +-----------------------------------+");
    displayData(theRAM);
    System.out.println("   +-----------------------------------+");

  }

  /**
   * Display [0..size-1] words of instruction memory. Translate back to ASM
   */
  private void displayProgram(InstructionMemory rom) {
    int lastValidIndex = rom.size() - 1;
    BitString[] bitstrings  = rom.rangeToBitString(zero, new BitString((short) lastValidIndex));

    for (int address = 0; address < bitstrings.length; address++) {
      Instruction instruction = new Instruction(bitstrings[address]);
      System.out.println(String.format("[%d] %s : %s",
                                     address, instruction,
                                       InstructionToASM.toASM(instruction)));
    }
  }


  /**
   * Display [0..size-1] words of memory.
   */
  private void displayData(Memory ram) {
    int lastValidIndex = ram.size() - 1;
    System.out.println(String.format("lastValidIndex = %d", lastValidIndex));

    BitString[] bitstrings  = ram.rangeToBitString(zero, new BitString((short) lastValidIndex));

    for (int address = 0; address < bitstrings.length; address++) {
      BitString bits = bitstrings[address];
      System.out.println(String.format("[%d] %s : %d",
                                     address, bits, bits.toShort()));
    }
  }
}
