package frozen;

import java.util.Scanner;

public class Receipt {
  class Node {
    FrozenTreat treat;
    Node next;
    public Node(FrozenTreat treat, Node next) {
      this.treat = treat;
      this.next = next;
    }
    public Node(FrozenTreat treat) {
      this(treat, null);
    }
  }

  /**
   * Read FrozenTreat description from the given scanner.
   *
   * In the case of a Sundae object, the base treat description is
   * parsed by a recursive call to this function. If a Sundae
   * decorates a Sundae, this can go down an arbitrary number of
   * levels.
   *
   * @param frozenTreatDescriptions Scanner, open on a _sequence_ of
   * valid FrozenTreat descriptions.
   *
   * @return a new FrozenTreat with an actual type as described in the
   * scanner.
   *
   * @note a valid description is a single line beginning with the
   * name of a FrozenTreat subclass followed by the description of its
   * field values:
   *   IceCream  <flavor>
   *
   *   FrozenYoghurt <flavor>
   *
   *   Sundae <topping>
   *   <base FrozenTreat>
   */
  public FrozenTreat nextFrozenTreat(Scanner frozenTreatDescriptions) {
    String line = frozenTreatDescriptions.nextLine();
    line = line.trim(); // remove leading/trailing whitespace
    Scanner lineScanner = new Scanner(line);
    String treatType = lineScanner.next(); // treatType cannot contain spaces
    if (treatType.equals("IceCream")) {
      String flavor = lineScanner.next();  // flavor cannot contain spaces
      return new IceCream(flavor);
    } else if (treatType.equals("Yoghurt")) {
      String flavor = lineScanner.next();  // flavor cannot contain spaces
      return new Yoghurt(flavor);
    } else if (treatType.equals("Sundae")) {
      String topping = lineScanner.next(); // topping cannot contain spaces
      FrozenTreat base = nextFrozenTreat(frozenTreatDescriptions);
      return new Sundae(topping, base);
    }
    return null;
  }

  Node head;
  public Receipt(Scanner fin) {
    head = null;
    while (fin.hasNextLine()) {
      FrozenTreat treat = nextFrozenTreat(fin);
      head = add(head, treat);
    }
  }

  private Node add(Node curr, FrozenTreat treat) {
    if (curr == null) return new Node(treat);
    curr.next = add(curr.next, treat);
    return curr;
  }

  public double total() {
    double t =  0.0;
    for (Node curr = head; curr != null; curr = curr.next)
      t += curr.treat.price();
    return t;
  }

  public String toString() {
    String receiptString = "";
    String separator = "";
    for (Node curr = head; curr != null; curr = curr.next) {
      receiptString = receiptString + separator + curr.treat;
      separator = "\n";
    }
    return receiptString;
  }
}
