package menu;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;

/**
 * Menu utility: displays and validates input for a text-based menu.
 *
 * Constructed with a source of text lines that is used to fill the
 * entries into the Menu. Each entry is a <command> which is what the
 * user types in to select that value; a <description> which is a short
 * description of what the command does; and a <help> text which is
 * shown when the user enters "?" to give them more information.
 *
 * display() and help() are public to permit client code to display
 * the menu or the help view of the contents of the menu to
 * stdout. They may not be very useful.
 *
 * contains(String) is a public boolean function, exposed to make outside
 * testing easier. It returns true if the given string is one of the
 * commands in the menu, false otherwise.
 */
public class BCLMenu
    implements Menu {

  class MenuEntry {
    String command;
    String description;
    String help;

    public MenuEntry(String command, String description, String help) {
      this.command = command;
      this.description = description;
      this.help = help;
    }

    public String menuText() {
      return String.format("%s - %s", command, description);
    }

    public String helpText(int indent) {
      String fmt = "%" + indent + "s - %s\n" +
          "%" + indent + "s   %s";
      return String.format(fmt, command, description, " ", help);
    }
  }

  private List<MenuEntry> entries;
  private int maxCommandWidth;

  public BCLMenu(Scanner sin) {
    entries = new ArrayList<>();
    maxCommandWidth = 0;

    while (sin.hasNextLine()) {
      String command = sin.nextLine().trim();
      String description = sin.nextLine().trim();
      String help = sin.nextLine().trim();

      add(command, description, help);
    }
  }

  public BCLMenu(InputStream in) {
    this(new Scanner(in));
  }

  private void add(String command, String description, String help) {
    maxCommandWidth = Math.max(maxCommandWidth, command.length());
    entries.add(new MenuEntry(command, description, help));
  }

  /**
   * Display the standard menu view of contents.
   *
   * @note public so that client code can test/print contents
   */
  public void display() {
    for (MenuEntry entry : entries) {
      System.out.println(entry.menuText());
    }
  }

  /**
   * Display the help view of contents.
   *
   * @note public so that client code can test/print contents
   */
  public void help() {
    for (MenuEntry entry : entries) {
      System.out.println(entry.helpText(maxCommandWidth));
    }
  }

  /**
   * Number of entries in the Menu.
   *
   * @note primarily for testing/debugging
   */
  public int size() {
    return entries.size();
  }

  /**
   * Test whether the given command is contained in the menu.
   *
   * @param couldBeACommand potential command to search for
   * @return true if couldBeCommand is a command in the menu, false
   * otherwise
   * @note case-sensitive match
   */
  public boolean containsCommand(String couldBeACommand) {
    boolean contains = false;
    for (MenuEntry entry : entries) {
      contains = contains || entry.command.equals(couldBeACommand);
    }
    return contains;
  }

  /**
   * Match user input, provided on the keyboard, with menu commands.
   *
   * Loop, printing the prompt and reading one line from keyboard as a
   * command.
   *
   * If command is empty, just loop
   * If command is a match with one in menu, return that command
   * If command is "?", show help text for menu and loop
   * If command is anything else, give an error message and loop
   *
   * @param prompt for display to user; permits same menu to be reused
   * in different contexts
   * @param keyboard Scanner from which input is read
   * @return the command entered by the user that matched in the menu
   * @note will NEVER return if called on an empty menu; matching is
   * case-sensitive.
   */
  public String match(String prompt, Scanner keyboard) {
    display();
    boolean matched = false;
    String command = "";

    while (!matched) {
      System.out.print(prompt);
      command = keyboard.nextLine().trim();

      if (command.isEmpty())
        continue;

      matched = containsCommand(command);

      if (!matched)
        if (command.equals("?"))
          help();
        else
          System.out.printf("Unknown menu command '%s'\n", command);
    }

    return command;
  }
}
