package menu;

import java.util.Scanner;

/**
 * Menu utility: displays and validates input for a text-based menu.
 *
 * Constructed with a source of text lines that is used to fill the
 * entries into the Menu. Each entry is a <command> which is what the
 * user types in to select that value; a <description> which is a short
 * description of what the command does; and a <help> text which is
 * shown when the user enters "?" to give them more information.
 *
 * display() and help() are public to permit client code to display
 * the menu or the help view of the contents of the menu to
 * stdout. They may not be very useful.
 *
 * containsCommand(String) is a public boolean function, exposed to
 * make outside testing easier. It returns true if the given string is
 * one of the commands in the menu, false otherwise.
 */
public interface Menu {

  /**
   * Display the standard menu view of contents.
   *
   * @note public so that client code can test/print contents
   */
  public void display();

  /**
   * Display the help view of contents.
   *
   * @note public so that client code can test/print contents
   */
  public void help();

  /**
   * Number of entries in the Menu.
   *
   * @note primarily for testing/debugging
   */
  public int size();

  /**
   * Test whether the given command is contained in the menu.
   *
   * @param couldBeACommand potential command to search for
   * @return true if couldBeCommand is a command in the menu, false
   * otherwise
   * @note case-sensitive match
   */
  public boolean containsCommand(String couldBeACommand);

  /**
   * Match user input, provided on the keyboard, with menu commands.
   *
   * Loop, printing the prompt and reading one line from keyboard as a
   * command.
   *
   * If command is empty, just loop
   * If command is a match with one in menu, return that command
   * If command is "?", show help text for menu and loop
   * If command is anything else, give an error message and loop
   *
   * @param prompt for display to user; permits same menu to be reused
   * in different contexts
   * @param keyboard Scanner from which input is read
   * @return the command entered by the user that matched in the menu
   * @note will NEVER return if called on an empty menu; matching is
   * case-sensitive.
   */
  public String match(String prompt, Scanner keyboard);
}
