package cpu;

import java.util.Queue;
import java.util.ArrayDeque;
import java.util.Scanner;

public class Simulation {
  // The queue holding user strings. An instance variable
  // in case it is needed across multiple methods in the object.
  private Queue<String> userEntries;

  /**
   * Default (and only) constructor for a Simulation (which is an app).
   *
   * Initialize the queue.
   */
  public Simulation() {
    userEntries = new ArrayDeque<String>();
  }

  /**
   * Standard in/out command loop for user to fill the queue
   *
   * Prompt shown to System.out; line read from System.in.
   * Loop halts on command "quit" or end-of-file.
   * Commands
   *   quit           - end loop
   *   enqueue <line> - add <line> to the queue
   *   dequeue        - remove and display item at queue head
   *   peek           - display item at queue head
   *
   * @note command processing improved to take any prefix of a valid
   * command rather than requiring the whole word
   */
  public void commandLoop() {
    Scanner keyboard = new Scanner(System.in);
    System.out.println("Q command: ");
    while (keyboard.hasNextLine()) {
      String line = keyboard.nextLine();
      if (line.isBlank())
        continue;

      Scanner lineScanner = new Scanner(line.strip());
      String cmd = lineScanner.next();
      if ("enqueue".startsWith(cmd)) {
        String whatToEnqueue = lineScanner.nextLine();
        whatToEnqueue = whatToEnqueue.strip();
        userEntries.add(whatToEnqueue);
        System.out.println(String.format("Enqueued \"%s\"", whatToEnqueue));
      } else if ("dequeue".startsWith(cmd)) {
        if (!userEntries.isEmpty()) {
          String whatWasDequeued = userEntries.poll();
          System.out.println(String.format("Dequeued \"%s\"", whatWasDequeued));
        } else {
          System.out.println("There is noting in the queue to dequeue.");
        }
      } else if ("peek".startsWith(cmd)) {
        if (!userEntries.isEmpty()) {
          String peekAtHead = userEntries.peek();
          System.out.println(String.format("Head of list = \"%s\"", peekAtHead));
        } else {
          System.out.println("There is noting in the queue to peek at.");
        }
      } else if ("quit".startsWith(cmd)) {
        break;
      }
    }
  }

  /**
   * Print out all elements in the queue, emptying it.
   * @return number of elements that were removed from the queue.
   */
  public int dumpLeftovers() {
    int leftoverElements = 0;
    while (!userEntries.isEmpty()) {
      String whatWasDequeued = userEntries.poll();
      System.out.println(String.format("\"%s\"", whatWasDequeued));
      leftoverElements++;
    }

    return leftoverElements;

   }

  /**
   * Load the queue with the commandLoop(), then dumpLeftovers().   *
   */
  public void run() {
    commandLoop();

    System.out.println("Leftover Elements");


    int leftoverElements = dumpLeftovers();

    System.out.println(String.format("There were %d leftover elements.", leftoverElements));
  }

  /**
   * Our standard main: construct a Simulation, call Simulation.run
   *
   */
  public static void main(String[] args) {
    Simulation simulation = new Simulation();
    simulation.run();
  }
}
